clear variables; close all; clc

% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%                                   INPUT
% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

movie               = 0;
%                   PLOTTING
grids               = 0;   % plots the Lagrangian and Eulerian grids
plot                = 1;   % main nappe emplacement plot
zoom                = 0;   % nappe emplacement plot: zoom on the thrust ramp
values              = 0;   % plots the variation of viscosity, 
                           % rediogenic heating, shear heating term

%                   MODEL CONDITIONS
Tdep_eta            = 1;   % T-dependent viscosity?
SH                  = 1;   % Shear heating?
H_depth             = 0;   % Depth-dependent radiogenic heating (exp decay)?

%                   STORE VALUES
% save temperature and viscosity values for each timestep?
T_and_eta           = 0;
% save viscosity values on the reference point stored for each timestep ?
T_and_eta_ref       = 0;   
% calculate and save Pe, Br values?
dimless_num         = 0;   

%                   TIME
tt                  = 300; % [s]
time_tot            = 5.1 * (3600*24*365.25*1e6);
%                   GEOMETRY
v_angle             = 15*pi/180; % rotation angle 
                                 % (for Lagrangian Velocity-grid) [rad]
ramp                = 15*pi/180; % theta angle [rad] == ramp angle
height              = 5*1e3;     % height of the ramp [m]

%                   PHYSICS
grad                = 0.01;     % Geothermal Gradient [K/m]
k                   = 2.5;      % Thermal Conductivity [W/m*K]
rho_c               = 2700*1e3; % Density * Heat Capacity [J/m^3*K]
DcT                 = k/rho_c;  % constant Thermal Diffusivity [m^2/s]
eta0                = 1e21;     % Viscosity [Pa*s]
Q                   = 2e5;      % Activation Energy [J/mol]
R_gas               = 8.314;    % Gas constant [J/K*mol]
H_rad_surf          = 5e-7;     % Rad Heat Prod at the surface [W*m^-3]
% length scale for rad heat prod decrease with depth [m]:
hr                  = 1e4;

% horizontal, prescribed, background velocity [m/s] (to obtain Pe=1):
% vx_bg               = 0.60649*1e-2 /(3600*24*365);
% horizontal, prescribed, background velocity [m/s]:
vx_bg               = 2 * 1e-2/(3600*24*365);

T_0_viscosity       = 500 + 273.15;             % reference temperature [K]

%                   NUMERICS
dt                  = 0.05;
nt                  = ceil(tt/dt);
nout                = 50;

%                   INITIAL
time                = 0;

% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%                               TEMPERATURE
% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

Lx_T                = 130*1e3;                                        % [m]
y_lowernappe_T      = 20*1e3;                                         % [m]
Ly_T                = 30*1e3 + y_lowernappe_T;                        % [m]
nx_T                = 100;
ny_T                = 50;
dx_T                = Lx_T/(nx_T-1);
dy_T                = Ly_T/(ny_T-1);
x_T                 = 0:dx_T:Lx_T;
y_T                 = 0:dy_T:Ly_T;
% 2D coordinate grid for Temperature (Eulerian > fixed):
[X_T Y_T]           = ndgrid(x_T,y_T);

x_vx_T              = -dx_T/2:dx_T:Lx_T+dx_T/2;
% 2D coordinate grid for Vx (Eulerian):
[X_Vx_T Y_Vx_T]     = ndgrid(x_vx_T,y_T);
y_vy_T              = -dy_T/2:dy_T:Ly_T+dy_T/2;
% 2D coordinate grid for Vy (Eulerian):
[X_Vy_T Y_Vy_T]     = ndgrid(x_T,y_vy_T);
% 2D coordinate grid for Velocity differentials:
[X_diffV Y_diffV]   = ndgrid(x_vx_T,y_vy_T);

VX_T                = zeros(size(X_Vx_T));
VY_T                = zeros(size(X_Vy_T));

% fault
xramp_T             = Lx_T/5;        % starting point (in x coordinate) [m]

% reference point
x_ref_T             = xramp_T + height*cot(ramp);
y_ref_T             = height + y_lowernappe_T;

% NUMERICS
dt_diff_T           = 1/4.1*min(dx_T,dy_T)^2/DcT;

% Initial Temperature profile
% set initial temperature on the reference point:
T(ceil(x_ref_T/dx_T),ceil(y_ref_T/dy_T)) = T_0_viscosity;
% temperature at the bottom of the Tgrid:
T_bottom_T          = T_0_viscosity + grad*(height+y_lowernappe_T);
% initial temperature (set on the Tgrid):
T_0                 = T_bottom_T*ones(size(Y_T)) + grad.*(-Y_T);

if H_depth == 1
    % depth-dependent heat production (exponential decay)
    [T, H_rad] = radiogenic_heat_production(H_rad_surf,hr,T_0,Y_T,nt,...
        dt_diff_T,time,DcT,rho_c,dx_T,dy_T); 
elseif H_depth == 0
    % constant radiogenic heat production
    [T, H_rad] = radiogenic_heat_production_constant(H_rad_surf,hr,T_0,...
        Y_T,nt,dt_diff_T,time,DcT,rho_c,dx_T,dy_T); 
end
T_INI       = T;

% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%                               KINEMATICS
% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

Lx_V                = 190*1e3;                                        % [m]
y_lowernappe_V      = 60*1e3;                                         % [m]
Ly_V                = 70*1e3 + y_lowernappe_V;                        % [m]
nx_V                = 150;
ny_V                = 75;
dx_V                = Lx_V/(nx_V-1);
dy_V                = Ly_V/(ny_V-1);
x_V                 = 0:dx_V:Lx_V;
y_V                 = 0:dy_V:Ly_V;
% 2D coordinate grid for Velocity (Lagrangian > moving):
[X_V Y_V]           = ndgrid(x_V,y_V); 

VX_V                = zeros(size(X_V));
VY_V                = zeros(size(Y_V));

% fault
xramp_V             = Lx_V/5;        % starting point (in x coordinate) [m]

% reference point
x_ref_V             = xramp_V + height*cot(ramp);
y_ref_V             = height + y_lowernappe_V;

% Minimize Eq. 11.8 to obtain gamma from the input ramp angle (theta)
% (Allmendinger et al., 2011)
options         = optimset('display','off');
gamma           = fzero('SuppeEquation',1.5,options,ramp);
% Compute slip ratio R (Eq. 11.8)
R               = sin(gamma - ramp)/sin(gamma);

% Geometry
% define fault geometry:
yf_V = faultgeometry(X_V,nx_V,x_V,xramp_V,height,ramp,y_lowernappe_V);
% define line 1 (between domains 1-2):
y1_V = -tan((pi-ramp)/2)*x_V + xramp_V*tan((pi-ramp)/2) + y_lowernappe_V;
% define line 2 (between domains 2-3):
y2a_V = (sin(ramp)/(cos(ramp)-R)) * x_V ...
    - (sin(ramp)/(cos(ramp)-R)) * (xramp_V+height*cot(ramp)) + height ...
    + y_lowernappe_V;  % from eq. 11.12 (used for stage a)
y2b_V = -tan((pi-ramp)/2)*x_V ...
    + (xramp_V+height*cot(ramp))*tan((pi-ramp)/2) + height ...
    + y_lowernappe_V;  %                (used for stage b)

% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%                                  ACTION
% ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

it = 0;
while time < time_tot
    it      = it+1;
    time    = time + dt;
    
    % VELOCITY FIELD
    for ix = 1:nx_V
        for iy = 1:ny_V
            [VX_V VY_V] = velocityfield(it,ix,iy,Y_V,X_V,VX_V,VY_V,...
                yf_V,xramp_V,ramp,height,vx_bg,R,dt);
        end
    end
    
    % ROTATION of the kinematic grid (Vgrid)
    % velocity field
    VX_V_rot = VX_V*cos(v_angle) - VY_V*sin(v_angle);
    VY_V_rot = VY_V*cos(v_angle) + VX_V*sin(v_angle);
    % grid
    X_V_rot  = X_V*cos(v_angle) - Y_V*sin(v_angle);
    Y_V_rot  = Y_V*cos(v_angle) + X_V*sin(v_angle);
    % reference point (reference point on the Vgrid)
    x_ref_V_rot = x_ref_V*cos(v_angle) - y_ref_V*sin(v_angle);
    y_ref_V_rot = y_ref_V*cos(v_angle) + x_ref_V*sin(v_angle);
    %Fault
    xf_V_rot   = x_V*cos(v_angle)  - yf_V*sin(v_angle);
    yf_V_rot   = yf_V*cos(v_angle) + x_V*sin(v_angle);
    
    % difference between the coordinates of the reference points
    dX_ref_rot = x_ref_V_rot - x_ref_T;
    dY_ref_rot = y_ref_V_rot - y_ref_T;
    % grid
    X_V_rot  = X_V_rot - dX_ref_rot;
    Y_V_rot  = Y_V_rot - dY_ref_rot;
    % reference point
    x_ref_V_rot = x_ref_V_rot - dX_ref_rot;
    y_ref_V_rot = y_ref_V_rot - dY_ref_rot;
    % fault
    xf_V_rot  = xf_V_rot - dX_ref_rot;
    yf_V_rot  = yf_V_rot - dY_ref_rot;
    
    % INTERPOLATION of the velocity field on the Temp grid (Eulerian)
    [VX_T]  = griddata(X_V_rot,Y_V_rot,VX_V_rot,X_Vx_T,Y_Vx_T,'nearest');
    [VY_T]  = griddata(X_V_rot,Y_V_rot,VY_V_rot,X_Vy_T,Y_Vy_T,'nearest');

    %======================================================================
    % PLOT grids & velocity arrows
    if grids == 1
        new_gridplot(x_ref_T,y_ref_T,X_V_rot,Y_V_rot,X_T,Y_T,...
            x_ref_V_rot,y_ref_V_rot,VX_V_rot,VY_V_rot,xf_V_rot,...
            yf_V_rot,X_Vx_T,Y_Vy_T,VX_T,VY_T,T);
    end
    %======================================================================
    
    % time-steps
    dt_adv_T  = min(dx_T/max(abs(VX_T(:))),...
        dy_T/max(abs(VY_T(:))))/10.1;   % time-step for advection
    dt        = min(dt_diff_T,dt_adv_T);
    
    % CALCULATIONS
    % Strainrates
    dVX_dx    = diff(VX_T,1,1)/dx_T;
    dVY_dy    = diff(VY_T,1,2)/dy_T;
    
    dVX_dy    = diff(VX_T,1,2)/dy_T;
    dVX_dy_c  = (dVX_dy(1:end-1,:) + dVX_dy(2:end,:))/2;
    dVX_dy_c  = (dVX_dy_c(:,1:end-1) + dVX_dy_c(:,2:end))/2;
    
    dVY_dx    = diff(VY_T,1,1)/dx_T;
    dVY_dx_c  = (dVY_dx(1:end-1,:) + dVY_dx(2:end,:))/2;
    dVY_dx_c  = (dVY_dx_c(:,1:end-1) + dVY_dx_c(:,2:end))/2;
    
    if Tdep_eta == 0
        % Viscosity constant
        eta = eta0*ones(size(T));
    elseif Tdep_eta == 1
        % Viscosity T-dep
        ex_fun              = (Q/R) .* (1./T-1./T_INI);
        ex_fun(ex_fun>0)    = 0;
        ex_fun(ex_fun<-3)   = -3;  % lower cutoff
        
        eta  = eta0 * exp( ex_fun );
        
        eta(eta>1e21) = 1e21;
        eta(eta<1e18) = 1e18;
    end
    
    % store viscosity values at the reference point:
    eta_ref(it)   = eta(ceil(x_ref_T/dx_T),ceil(y_ref_T/dy_T));
    
    if SH == 0
        H_shearheat = zeros(size(dVX_dx(2:end-1,2:end-1)));
    elseif SH == 1
        % Shear Heating
        D_XY        = (dVX_dy_c(2:end-1,:) + dVY_dx_c(:,2:end-1))/2; % def
        D_YX        = D_XY;
        Tau_XX      = 2*eta(2:end-1,2:end-1).*dVX_dx(2:end-1,2:end-1);
        Tau_YY      = 2*eta(2:end-1,2:end-1).*dVY_dy(2:end-1,2:end-1);
        Tau_XY      = 2*eta(2:end-1,2:end-1).*D_XY;
        Tau_YX      = Tau_XY;
        
        H_shearheat = Tau_XX .* dVX_dx(2:end-1,2:end-1) + ...
            Tau_YY .* dVY_dy(2:end-1,2:end-1) + ...
            Tau_XY .* D_XY + Tau_YX .* D_YX; % [Pa/s] = [W/m^3]
    end
    
    % DIFFUSION-ADVECTION of Temp
    qx    = - DcT*diff(T(:,2:end-1),1,1)/dx_T;
    qy    = - DcT*diff(T(2:end-1,:),1,2)/dy_T;
    dTdt  = (diff(-qx,1,1)/dx_T +  diff(-qy,1,2)/dy_T) ...
        - VX_T(2:end-2,2:end-1).*diff(T(1:end-1,2:end-1),1,1)/dx_T ...
        - VY_T(2:end-1,2:end-2).*diff(T(2:end-1,1:end-1),1,2)/dy_T ...
        + H_shearheat/rho_c + H_rad(2:end-1,2:end-1)/rho_c;
    
    dTdt_ad = - VX_T(2:end-2,2:end-1).*diff(T_INI(1:end-1,2:end-1),1,1)/...
        dx_T - VY_T(2:end-1,2:end-2).*diff(T_INI(2:end-1,1:end-1),1,2)/dy_T;
    
    T(2:end-1,2:end-1)  = T(2:end-1,2:end-1) + dTdt*dt;
    T(1,:)              = T(2,:);
    T(end,:)            = T(end-1,:);
    T_INI(2:end-1,2:end-1)  = T_INI(2:end-1,2:end-1) + dTdt_ad*dt;
    T_INI(1,:)              = T_INI(2,:);
    T_INI(end,:)            = T_INI(end-1,:);
    
    % Calculate the coordinates of the points in the Lagrangian grid 
    %                                                (ADVECTION GRID)
    X_V(2:end,:) = X_V(2:end,:) + VX_V(2:end,:) * dt;
    Y_V          = Y_V          + VY_V          * dt;
    
    % REFINEMENT of Lagrangian mesh
    horizontal_distance_V     = X_V(2,(ceil(y_lowernappe_V/dy_V)+1)) - ...
        X_V(1,(ceil(y_lowernappe_V/dy_V)+1));
    if horizontal_distance_V  > 2*dx_V
        X_V_c    = (X_V(1,:) + X_V(2,:))/2;
        X_V      = [X_V(1,:);  X_V_c;     X_V(2:end,:)];
        Y_V      = [Y_V(1,:);  Y_V(1,:);  Y_V(2:end,:)];
        VX_V     = [VX_V(1,:); VX_V(1,:); VX_V(2:end,:)];
        VY_V     = [VY_V(1,:); VY_V(1,:); VY_V(2:end,:)];
    end
    
    %======================================================================
    % PLOT
    if mod(it,nout)==0
        
        if plot ==1
            new_plot(T_INI,X_T,Y_T,T,X_Vx_T,Y_Vx_T,VX_T,VY_T,X_V_rot,...
                Y_V_rot,VX_V_rot,VY_V_rot,xf_V_rot,yf_V_rot,...
                H_shearheat,eta,Lx_T,Ly_T,y_T,time,dX_ref_rot,...
                dY_ref_rot,x_ref_T,y_ref_T,x_ref_V_rot,y_ref_V_rot);
        end
        % ZOOM ON THE RAMP
        if zoom == 1
            new_plot_zoom(X_T,Y_T,T,X_Vx_T,Y_Vx_T,VX_T,VY_T,X_V_rot,...
                Y_V_rot,VX_V_rot,VY_V_rot,xf_V_rot,yf_V_rot,...
                H_shearheat,Lx_T,Ly_T,y_T,time)
        end
        % PLOT values
        if values ==1
            plot_values(Y_T,T,eta,H_rad,rho_c,H_shearheat);
        end
        
        %==================================================================
        % Create IMAGES for the movie
        if movie == 1
            print(['./faultbendfold_' num2str(it,'%05d')],'-dpng','-r300')
        end
        %==================================================================
        
    end
    
    %======================================================================
    % SAVE values
    if T_and_eta == 1
        mkdir Temp_Visc
        [a, MSGID] = lastwarn();
        warning('off', MSGID)
        save(['./Temp_Visc/Temp_Visc' num2str(time./(3600*24*365*1e6),2)...
            '.mat'],'T','eta')
    end
    %======================================================================
    
    if dimless_num == 1
        % ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        % CALCULATION OF DIMENSIONLESS NUMBERS (Duprat-Oualid et al., 2015)
        % ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
        
        Pe(it) = height*vx_bg/DcT;          % Peclet Number  (Adv     vs Diff)
        Br(it) = (eta_ref(it)*(vx_bg^2))/...% Brinkman Number(Prod_sh vs Diff)
            (k*T(ceil(x_ref_T/dx_T),ceil(y_ref_T/dy_T)));   
        
        R_adv(it)   = Pe(it) /(Pe(it)+1+Br(it));  % Advection  contribution
        R_diff(it)  =  1 /(Pe(it)+1+Br(it));      % Diffusion  contribution
        R_pro(it)   = Br(it) /(Pe(it)+1+Br(it));  % Production contribution
        
        % Store Pe, Br, R dimensionless numbers
        PeBr(it,:)      = [Pe(it) Br(it) time./(3600*24*365*1e6)];
        R_numbers(it,:) = [R_adv(it) R_diff(it) R_pro(it) ...
            time./(3600*24*365*1e6)]; % time in Myr
    end
    
end

%==========================================================================
% SAVE values

if dimless_num == 1
    writematrix(R_numbers,'R_numbers.csv')
    writematrix(PeBr,'PeBr.csv')
end

if T_and_eta_ref == 1
    mkdir Temp_Visc_ref
    save(['./Temp_Visc_ref/Temp_Visc_ref' num2str(it) '.mat'],'eta_ref')
end
%==========================================================================

